/*  Copyright 2023 Dieter Holzhäuser  E-Mail: dieter.holzhauser@gmail.com
  
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published serbuf
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mtos.h"

void inittimer () {
  TCCR2A   = 0x00;      // Bit       1 0
                        //           0 0  Normal Mode
  TCCR2B   = 0x04 ;     // Bit   3 2 1 0
                        //         1 0 0  Prescaler 1/64
                        //       0        Normal Mode
  TCNT2 = 6;            //Start Timer2, 256-6=250 ticks to overflow
  TIMSK2 |= (1 <<  TOIE2);    //enable Interrupt on overflow 
}
ISR(TIMER2_OVF_vect) {
  TCNT2 = 6;             //Restart
  os.tinterrupt ();
}
/////////////////////////////////////////////////////////////////
/////////////////////////Multitasking Kernel  ///////////////////

      //Task state elements (masks)
#define  WAITING           7    //00000111
#define  SEMABIT           8    //00001000
#define  TIMERBIT         16    //00010000
#define  PINBIT           32    //00100000

///////////////linked list functions 
                       //append task to linked list ready/wait
void Mtos::append (unsigned char task, unsigned char * ptop){
  unsigned char i, prec = 0;
  if (ptop == &readytop) tasklist[task].state = READY;
  if (*ptop == NIL)
    *ptop = task;                       //new
  else {
    for (i = *ptop ; i != NIL; i = tasklist[i].next)  prec = i;
    tasklist[prec].next = task;         //append
  }
  tasklist[task].next = NIL ;
}
                          //delete task from from linked list ready/wait
void Mtos::del (unsigned char task, unsigned char* ptop ) {
  unsigned char i , prec = 0;
  for (i = *ptop; i != NIL; i = tasklist[i].next) {
    if ( i == task) {
      if (i == *ptop)
        *ptop = tasklist[i].next;                   //from top position
      else tasklist[prec].next = tasklist[i].next;  //other position
    }
    prec = i;
  }
}

void Mtos::runner (void) {     //"motor" of program,  call runner in setup, avoid loop() 
  while (1) {
    if ( readytop != NIL  ) {
      DI
      runtask = readytop;
      readytop = tasklist[runtask].next;
      tasklist[runtask].state = RUNNING;
      EI
      tasklist[runtask].mainpfunc ();            //call task function
      DI
      if (tasklist[runtask].state == RUNNING) {  //task function returned
        tasklist[runtask].state = TERMINATED;    //without mt function call
        del (runtask, &waittop);
      }
      EI
    }
    else  idlecycs++;              //count idle cycles
  }
}

void Mtos::start (unsigned char task, void (*pfunc) (void) ) {
  DI
  if (tasklist[task].state == TERMINATED)   {
    tasklist[task].mainpfunc = pfunc;
    append (task , &readytop);
  }
  EI
}

void Mtos::stop (unsigned char task, void (*pfunc) (void) ) {   
  if (tasklist[task].state != TERMINATED) {
    if ( pfunc ) pfunc ();     //function to create a defined state or zero
    DI
    del ( task, &readytop);
    del (task, &waittop);
    tasklist [task].state = TERMINATED;
    EI
  }
}

void Mtos::tinterrupt() {
  clock++;
  if (idlecycs < idlemin ) idlemin = idlecycs;     //save idle minimum
  idlecycs = 0;
  if (++timercycs >= ONESECOND ) {
    timercycs = 0;
    bclock++;
    idle = idlemin; 
    idlemin = 10000;
  }
  unsigned char i ;
  for (i = 0; i != MAXTASKS; i++ ) {
    if (tasklist[i].state & PINBIT ) {            //detect WAITING_B input pins
      unsigned char pinact =  (unsigned char) RPIN ( tasklist[i].pinnr );
      if (  pinact !=   tasklist[i].pinold ) {
        tasklist[i].pinold   =  pinact;
        if ((pinact  &&  tasklist[i].edge) ||         //rising edge or
              (!pinact &&  ! tasklist[i].edge)) {     //falling edge
          tasklist[i].mainpfunc = tasklist[i].altpfunc;
          append (i , &readytop);                        //set event
          continue;
        }
      }
    }
    if (tasklist[i].state & TIMERBIT ) {            //Manage task timer
      if (--tasklist[i].timer == 0) {
        if (tasklist[i].state & SEMABIT)  del (i, &waittop );
        append (i , &readytop);                     //set event
      }
    }
  } 
}

bool Mtos::timeinrange (long from, long to) {      
  if (to < from )             
    return ! (bclock >= to && bclock < from) ;  //midnight in range             
  else return (bclock < to && bclock >= from);  //midnight not in range         
}

long Mtos::get_clock () {
  return clock ;
}

long Mtos::get_timer (unsigned char task   ) {
   return tasklist[task].timer;
}

unsigned char Mtos::get_taskstate (unsigned char task) {
  return tasklist[task].state;
}

bool Mtos::is_stop (unsigned char task) {
  return tasklist[task].state == TERMINATED;
}

unsigned char Mtos::get_tasksema (unsigned char task) {
  unsigned char n;
  if (tasklist[task].state == WAITING+SEMABIT)
    n = tasklist[task].sema;
  else n = NIL;
  return n;
}

unsigned char Mtos::get_runtask (void) {
  return runtask;
}

unsigned char Mtos::issema (unsigned char sema ) {
  unsigned char n;
  n = semalist [sema];
  if (n) return 1; else return 0;
}

/////mt functions
void Mtos::mtcoop ( void (*pfunc) (void)   ) {
  DI
  if (tasklist[runtask].state == RUNNING) {
    tasklist[runtask].mainpfunc = pfunc;
    append (runtask, &readytop);
  }
  EI
}
void Mtos::mtsema (unsigned char sema , void (*pfunc) (void) ) {
  DI
  if (tasklist[runtask].state >= RUNNING) {
    if ( semalist[sema] > 0 ) {
      semalist[sema] = 0;
      tasklist[runtask].mainpfunc = pfunc;
      append (runtask , &readytop);     //state READY, like mtcoop
    }
    else {
      tasklist[runtask].sema = sema;
      tasklist[runtask].altpfunc = pfunc;
      tasklist[runtask].state |= WAITING + SEMABIT;
      tasklist[runtask].state &= ~PINBIT;
      append (runtask , &waittop);
    }
  }
  EI
}
void Mtos::mtdelay ( long t , void (*pfunc) (void)  )  {
  DI
  if (t <= 0) 
    mtcoop (pfunc);
  else {
    if ( tasklist[runtask].state >= RUNNING) {
      tasklist[runtask].timer = t;
      tasklist[runtask].mainpfunc = pfunc;
      tasklist[runtask].state |= WAITING + TIMERBIT;
    }
  }
  EI
}
void Mtos::mtpin ( unsigned char pinnr, unsigned char edge, void (*pfunc) (void) ) {
  DI
  if ((tasklist[runtask].state >= RUNNING)) {
    tasklist[runtask].pinnr  = pinnr;
    tasklist[runtask].edge  = edge;
    tasklist[runtask].pinold  =  (unsigned char) RPIN (pinnr);
    tasklist[runtask].altpfunc = pfunc;
    tasklist[runtask].state |= WAITING + PINBIT;
    tasklist[runtask].state &= ~SEMABIT;
  }
  EI
}

/////Sema functions
void Mtos::signalisr (unsigned char sema ) {
  unsigned char i , prec = 0;
  for (i = waittop; i != NIL; i = tasklist[i].next) {
    if  (tasklist[i].sema == sema) {         //take out of waitlist
      if (i == waittop)
          waittop = tasklist[i].next;                   // top position
      else tasklist[prec].next = tasklist[i].next;      // other position
      tasklist[i].mainpfunc = tasklist[i].altpfunc;
      append (i , &readytop);
      return;
    }
    prec = i;
  }
  semalist [sema] = 1;
}

void Mtos::signal (unsigned char sema ) {
  DI
  signalisr (sema);
  EI
}

void Mtos::clearsema (unsigned char sema ) {
  semalist [sema] = 0;
}
/////
void Mtos::initmt () {           // call in Arduino setup ()
  unsigned char i ;
  for ( i = 0; i < MAXTASKS; i++) tasklist[i].state = TERMINATED;
  for ( i = 0; i < MAXSEMAS; i++) semalist[i] = 0; 
  inittimer ();
  EI
}
/////
Mtos os;

////////////////////////Serial Input///////////////////////////

///////////////////Class Sui  Support of Task SUI 
char Sui::test_cmd() {         //command in list?
  PR (sr.cmd);                 //print command          
  PR (" ");
  char mcmd [LCMD];
  for (char m = 0; m < maxcmds ; m++) {    //find command
    char* vpt = (char*) cmds[m].pt; 
    for (int n = 0; n < LCMD; n++) {       //copy ROM saved command to mcmd
      mcmd [n] =  pgm_read_byte(vpt++); 
      if ( mcmd [n] == ' ' || mcmd [n] == 0 ) {
        mcmd [n] = 0 ; 
        break;
      }
    }  
    if (strcmp (sr.cmd , mcmd) == 0)  return m;  //compare    
  } 
  mprint (inval_cmd);          //command not in list
  return -1;
}   

char Sui::num_ch () { 
  ch = Serial.read ();  
  if (ch == CR) ch = LF;   
  switch (prcode) {       //character is processed depending on prcode 
    case -1:              //error
    case -2: 
    case -3:
      inum = prcode;
      break;
 ///////////  
    case 0:               //detect and check first numeric character 
      if (ch == ' ' )     //ignore leading blanks 
        ;                            
      else if (ch == '-') {  
             if (minus) prcode = -1;      //illegal character                
             minus = true;
           }
           else if (ch == '#')      
                  prcode =  2;
                else if (ch >= '0' && ch <= '9') {                          
                       num = ch - '0';    //initialize transformation 
                       prcode = 1;   
                     } 
                     else if (ch == LF) 
                            break;                       
                          else prcode = -1;     //illegal character                                                             
      return 0;      //next character
////////////
    case 1:                          //check next numeric character
      if (ch >= '0' && ch <= '9') {  //continue transformation 
          if (num <= 214748364) {    //least significant decimal
            num = num * 10;
            num = num + ch - '0';
            if ( num < 0 ) prcode = -2;      //out of range       
          }  
          else prcode = -2;                  //out of range             
          if (deci) deci++;        
      }
      else if (ch == '.')  {         //decimal point detected 
             if (deci) 
               prcode = -1;          //illegal character   
             else deci = 1;                                    
           }
           else if (ch == ' '|| ch == LF) { //blank as num separator or end of input 
                  if (minus) num = -num;                     
                  lnum [inum++] = num; 
                  if (! (deci == 0 || deci == MAXDECI) )
                    prcode = -3;        //only point and two decimal digits accepted
                  else {  
                    if (ch == LF || inum == MAXLNUM)  //end of input   
                      break;
                    else return 2;          //next number  
                  }
                }
                else if (ch == '/')
                       ;
                     else prcode = -1;      // illegal character                              
      return 0;                         //next character
//////////////////             
    case 2:                       //character # detected (no number)
      lnum [inum++] = NONB; 
      if (ch == LF || ch == ' ') {
          if (ch == LF || inum == MAXLNUM)  //end of input
            break;
          else return 2;      //next number  
      }
      else prcode = -1;       //illegal character
      return 0;               //next character
  }
  return 1;                   //end of num detection is end of input
}

char Sui::read_cmd () {     //read command from serial input buffer
  char n = 0;
  inum = 0;
  while (1) { 
    char c = Serial.read ();  
    if (c == ' ') {         //end of command by blank or leading blank
      if (n > 0)  {         
        if (n == LCMD)      //command too long, error
          return 0;         
        else {
          cmd[n] = 0;       //set end character
          return 2;         //continue by detecting numbers  
        }  
      }                     
      else continue ;       //ignore leading blank
    } 
    else if (c == LF || c == CR ) {  //end of command, command only 
           if (n == LCMD)     //command too long, error
             return 0;       
           else {
             cmd[n] = 0;      //set end character
             return 1;        //continie by calling command function     
           }      
         }   
         else {
           cmd[n] = c;        //save command character       
           if ( n < LCMD ) n++;   
         }                       
  }
}

char Sui::clear () {     //clear serial input buffer
  ch = Serial.read ();  
  if (ch == -1) {
    nbufold = 0;
    return 0;            //continue by mtcoop (sui_home); 
  }    
  else return 1;         //continue by mtcoop (sui_clear);   
}

void Sui::init_next_num () {
  prcode = 0;  
  minus = false; 
  deci  = 0;
}

char Sui::home () {            
  nbuf = Serial.available();
  if (nbuf >= 64)              //buffer overflow, input aborted
    return 0;                  //continue by sui_clear      
  else if (nbuf == nbufold){   //buffer stable
         if (nbuf)               
           return 1;           //input done
         else return 2;        //no input,  repeat reading buffer 
       }
       else {                  //input is made
         nbufold = nbuf;
         return 2;             //repeat reading buffer 
       }    
} 

void sui_home ();
void Sui::initserinp (){ 
  os.start (SUI, sui_home); 
} 
/////////////
Sui sr;

///////////////////Task SUI    Task functions
void sui_next_num ();

void sui_clear (){             
  switch (sr.clear() ) {             //clear serial input buffer 
    case 0: os.mtcoop (sui_home); break;
    case 1: os.mtcoop (sui_clear); 
  } 
}

void sui_menu () {          //each line delayed  
  PR (' ');
  mprint ( cmds[sr.imenu].pt );
  PRln ();
  if (++sr.imenu == maxcmds)
    os.mtcoop (sui_clear);
  else os.mtdelay (20, sui_menu);
}  
    
void sui_num_ch () {            
  switch (sr.num_ch() ) {            //process a character
    case 0:   os.mtcoop (sui_num_ch); break;  //next character (inner loop, explicite cooperation)
    case 1:   os.mtcoop (cmds[sr.icmd].p); break;    //let call command function of application
    case 2:   os.mtcoop (sui_next_num);       //next number
  }
}

void sui_next_num () {          
  sr.init_next_num();                   //initialize number detection by num_ch
  os.mtcoop ( sui_num_ch );  
}

void sui_cmd (){ 
  sr.inum = 0;
  switch ( sr.read_cmd() ) {            //read command
    case 0:   mprint (inval_cmd);       //command error
              os.mtcoop (sui_clear);
              break;
    case 1:   if ( sr.cmd[0] == 0){        //command only / LF menu?
                mprint (LF_menu);
                sr.imenu = 0;
                os.mtdelay (20, sui_menu); 
              }
              else {
                sr.icmd = sr.test_cmd() ;          //command only, test command
                if ( sr.icmd  == -1)   
                  os.mtcoop (sui_clear);  
                else os.mtcoop (cmds[sr.icmd].p);  //let call command function of application
              }
              break;
    case 2:   sr.icmd = sr.test_cmd() ;            //test num command    
              if (sr.icmd == -1)   
                os.mtcoop (sui_clear);
              else  os.mtcoop (sui_next_num);  
  }
}

void sui_home (){              
  switch (sr.home() ) {            //detect stable serial input
    case 0:  os.mtcoop (sui_clear); break;
    case 1:  os.mtcoop (sui_cmd);  break;
    case 2:  os.mtdelay (10, sui_home);      
  }
}  

///////////////////////EEMEM//////////////////
//////////////read
void eeread (unsigned int eeadr, unsigned char size, void * pvar ) {
  unsigned char n;
  for ( n = 0; n < size; n++) {
    DI
    eeadr = eeadr + n;
    EEARL = eeadr % 256;    //Set up address register
    EEARH = eeadr / 256;
    EECR |= (1<<EERE);      //Start eeprom read by writing EERE
    ((unsigned char *) pvar) [n] = EEDR;   //data from data register
    EI
  }
  os.signal (EESEM);
}
///////////write
void (*eeretf)(void);      //save task function to continue calling task
unsigned char *spvar;      //save pointer to first byte of var
unsigned char  ssize;      //save size of var
unsigned int   seeadr;     //save EEMEM adress
unsigned char  een;        //counter

void ee1 () {                   
  if ( !(EECR & (1<<EEPE) ) ) {  // completion of previous write
    if (een == ssize ) {         //end of eewrite
      os.signal (EESEM);
      os.mtcoop (eeretf);
      return;                
    }
    DI
    EECR = (0<<EEPM1)|(0<<EEPM0);   //Set Programming mode
    seeadr = seeadr + een;          //Set up address registers
    EEARL = seeadr % 256;
    EEARH = seeadr / 256;
    EEDR =  spvar[een];          //Set up data register
    EECR |= (1<<EEMPE);          //Write logical one to EEMPE
    EECR |= (1<<EEPE);           //Start eeprom write by setting EEPE
    EI
    een++;
  }
  os.mtdelay (1, ee1);
}
                       //init subtask eewrite
void eewrite (void * pvar, unsigned char size, unsigned int eeadr, void (*retf)(void) ) {
  eeretf = retf;
  spvar = (unsigned char *) pvar;
  seeadr = eeadr;
  ssize = size;
  een = 0;
  os.mtcoop (ee1);
}

////////////////////////
///////////Print ROM 
void mprint (const char str[] ) {    
  char c;                           
  while((c = pgm_read_byte(str++)))  PR (c);  
}  
////////time
long trtime (unsigned long n) {      //transform time
  if (n % 100 > 59 || (n / 100) % 100 > 59 || n / 10000 > 23 ) return SECDAY;
  return ( n % 100 + ((n / 100) % 100 ) * 60 + n / 10000 * 3600 );
}
void pr (int m){
  if ( m < 10) PR ("0");
  PR (m);
}
void prtime (long n) { 
  if (n < 0) n = -n;
  n = n  % SECDAY;  
  pr ( n / 3600 );
  pr ( (n / 60)% 60 );
  pr ( n % 60 );
  PR ("c");
}

///////////class  Tnum         
Tnum::Tnum (long nb0, long nb1) {
  nb[0] = nb0;
  nb[1] = nb1;
}

void Tnum::outp ( const char * t1 ){
  char n = 0;
  while (t1[n]){
    if (t1[n] != '$')
      PR (t1[n] ); 
    else  switch (t1[++n]) {
////number output format: i(I) long, f(F) float, t(T) hhmmss
            case 'i':  if (nb[0] == NONB )
                         PR ("#");
                       else PR (nb[0]);  
                       break;
            case 'I':  if (nb[1] == NONB )
                          PR ("#");
                       else PR (nb[1]);           
                       break;
            case 'f':  PR ( (float)nb[0] / 100 ); break;
            case 'F':  PR ( (float)nb[1] / 100 ); break;
            case 't':  prtime (nb[0] ); break;
            case 'T':  prtime (nb[1] ); break;
////special outputs
            case 'g':  PR (">> "); break;
            case 's':  PR ("<< "); break;
            case 'n':  PRln (); break;
            case '$':  PR ("$");  break;
          }
    n++;
  }
}


