/*  Copyright 2023 Dieter Holzhäuser  E-Mail: dieter.holzhauser@gmail.com
  
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published serbuf
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef MTRTOS_H_INCLUDED
#define MTRTOS_H_INCLUDED

#include <Arduino.h>

//Multitasking Realtime Operating System
//realtime compatible serial input
//realtime compatible use of EEMEM

// short alternatives
#define RPIN    digitalRead
#define WPIN    digitalWrite
#define PR      Serial.print
#define PRln    Serial.println
#define DI      SREG = SREG & ~0x80;    //Disable Interrupt
#define EI      SREG = SREG | 0x80;     //Enable Interrupt

//Tasks
#define MAXTASKS  20      //possible application tasks: 0 to MAXTASKS-2
#define SUI      MAXTASKS-1    //Library Task to manage Serial User Input
//Semaphores                             
#define  MAXSEMAS   20    //possible application semas: 0 to MAXSEMAS-2   
#define  EESEM      MAXSEMAS -1  //Library Sema: EEMEM access mutual exclusion                          
                                         
////////////////////////////////////////////////////////////
////////////////////Multitasking Kernel  ///////////////////
//call in setup:   os.initmt (...)    library init
//                 ....
//                 os.start (...) 
//                 ...
//                 os.runner ()       as last statement, 
//avoid  loop (), if not, it is never called

#define  ONESECOND  1000    //ms,  idle analyse period

/////Task states, do not change
#define READY        0
#define TERMINATED   5
#define RUNNING      6
#define WAITING_S    15     //Semaphore (Sema)
#define WAITING_T    23     //Time
#define WAITING_B    39     //Input pin bit edge
#define WAITING_TS   31     //Both Time or Sema
#define WAITING_TB   55     //Both Time or Input pin bit edge

#define  NIL   255

class Mtos {
  private:
  long clock =  0;                 //ms since start of program 
  unsigned char runtask = NIL;    //currently running task
  int idlecycs = 0;               //counter of runner idle cycles, see runner
  int idlemin = 10000;            //minimal number of idle cycles per ms in previous ONESECOND period
  int timercycs = 0;              //makes ONESECOND period, see ISR
  struct {                        
    unsigned char next;
    unsigned char state;
    void (*mainpfunc)(void) ;
    long timer;
    void (*altpfunc)(void) ;
    unsigned char sema;
    unsigned char edge;
    unsigned char pinold;
    unsigned char pinnr;
  } tasklist[MAXTASKS] ;      //Array of task data structures          

  unsigned char semalist[MAXSEMAS];   //Array of sema data
           //two linked lists are embedded in tasklist by component next
  unsigned char readytop = NIL;  //linked list readylist top index
  unsigned char waittop = NIL;   //linked list waitlist  top index         
  /////////
  void del (unsigned char task, unsigned char* ptop ) ;    //deactivate  task from list        
  void append (unsigned char task, unsigned char * ptop);  //activate task in list     
                                                  
  public:
  int idle;              //save idlemin, see idlemin           
  long bclock = 0;       //adjustable counter of seconds (basic clock)
  
  bool timeinrange (long from, long to); //true, if bclock is in range from-to| 
  long get_clock ();                     //can become negative
  long get_timer (unsigned char task);
  bool is_stop (unsigned char task); 
  unsigned char get_taskstate (unsigned char task);
  unsigned char get_tasksema (unsigned char task);
  unsigned char get_runtask (void); 
  unsigned char issema (unsigned char sema );  
  
  void runner ();       //endless, task functions are called ("motor" of program)
  void start  (unsigned char task, void (*pfunc) (void) ); //Start of task by task function pfunc
  void stop (unsigned char task, void (*pfunc) (void) );   //Call function pfunc and stop task
  
  void mtcoop (void (*pfunc) (void) );
  void mtsema (unsigned char sema , void (*pfunc) (void) );
  void mtdelay ( long t , void (*pfunc) (void)  );     // t ms timer
  void mtpin ( unsigned char pinnr, unsigned char edge, void (*pfunc) (void) ) ;
  
  void tinterrupt ();           //code of Interrupt Handler Timer2     
  void signalisr (unsigned char sema);
  void signal (unsigned char sema);
  void clearsema (unsigned char sema);
  void initmt  ();     
};
extern Mtos os;     //one object only

////////////////////////////SERIAL INPUT/////////////////////////
//Format: (cmd (number) (number)) LF      ( ) optional, ~ equivalent
//cmd:    command string 1 to 6 characters
//number: # || 123 || -123 || 001234 || 1.23 (~ 123) || -1.23 (~ -123)
//        digits 1234 as example,   # ~  NONB (no number), 
//        001234 used as time format hhmmss, 1.2 or 1.234 -> error,  
//LF:     Enter  
   
#define  NONB  -2147483648        //Numeric equivalent of exceptions
#define  LF    10                 //Line feed
#define  CR    13                 //Carrige return

#define LCMD    9          //Length of command string
#define MAXLNUM 2           //Max num data
#define MAXDECI 3           //Decimal digits + point 
#define SECDAY  86400

//short alternatives
#define IA sr.lnum[0]      //first number entered
#define IB sr.lnum[1]      //second number entered
#define OA nb[0]           //first number of Tnum object
#define OB nb[1]           //second number of Tnum object

void initserinp ();

class Sui {
  private:
  char deci;
  bool minus;
  long num;
  char ch; 
  char prcode;          //num processing code 
  int nbuf;
  int nbufold = 0;    
  public:
  //data interface  
  char inum;            //number of numeric data entered, inum == 0 means no num 
  long lnum [MAXLNUM];  //array of numeric data entered
  char imenu;
  char cmd[LCMD];      //command  
  char icmd; 
  /////////
  void initserinp ();
  //support of task functions 
  char num_ch ();
  char read_cmd ();
  char clear ();
  void init_next_num ();
  char home ();
  char test_cmd ();
};

extern Sui  sr;       //one object only

//Transform user inputs (commands) into activities by
//completion of library task SUI into the application 
//   (it's easier if data are not a member of class Sui)
struct cmdstruct {
  const char * pt;   //ROM Text  (commamd and descriptions)
  void (*p)();       //assigned command function (is SUI task function) 
};
extern struct cmdstruct cmds [];
extern char maxcmds;
//////end of completion declarations

//ROM text
const char num_error [] PROGMEM = "number error\n";
const char inval_cmd [] PROGMEM = "invalid command, LF menu\n";
const char range_error [] PROGMEM = "out of range\n";
const char num_missed [] PROGMEM = "number missed\n";
const char LF_menu [] PROGMEM = "LF menu\n";

////no class member functions
void mprint (const char str[]);     //Print ROM
void pr (int m);                    //aux prtime
void prtime (long n );              //print time, n(sec) -> print hhmmss
long trtime (unsigned long n);      //transform time,  hhmmss(n) -> ret sec

///////
void sui_clear();
void sui_menu();
////////class Tnum  save and print numbers
class Tnum {
  public:
  long nb[2];
  Tnum (long nb0, long nb1);
  void outp ( const char * t1 );    //output as format string
};


//////////////////////EEMEM/////////////////
//EEMEM adresses 0.. (eeadr) have to be defined. Compiler should not control them. 
//call eewrite as subtask, EESEM is exclusion semaphore 

void eeread (unsigned int eeadr, unsigned char size, void * pvar );
//if call can take place during write operation, call mtsema (EESEM,...) first.

//eewrite is a subtask
void eewrite (void * pvar, unsigned char size, unsigned int eeadr, void (*retf)(void) ) ;
//if call can take place during write operation, call mtsema (EESEM,...) first.

#endif    
 


